'use client';

import React, { useState, useEffect } from 'react';
import { useRouter, useParams } from 'next/navigation';
import {
  User,
  ArrowLeft,
  Edit,
  Mail,
  Phone,
  Building2,
  Shield,
  Calendar,
  Activity,
  TrendingUp,
  Info
} from 'lucide-react';
import Link from 'next/link';
import type { AgencyUser } from '@/types';
import { agencyUserApi, departmentApi } from '@/services/api';

export default function UserDetailPage() {
  const router = useRouter();
  const params = useParams();
  const userId = params?.id as string;

  const [user, setUser] = useState<AgencyUser | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (userId) {
      fetchUser();
    }
  }, [userId]);

  const fetchUser = async () => {
    try {
      setLoading(true);

      // Fetch user from API
      const data = await agencyUserApi.getAgencyUser(userId);

      // Convert API response to frontend format
      const userWithDates: AgencyUser = {
        id: data.id,
        firstName: data.first_name,
        lastName: data.last_name,
        email: data.email,
        phone: data.phone,
        avatar: data.avatar,
        departmentId: data.department_id,
        department: {
          id: data.department_id,
          name: data.department_name || '',
          color: data.department_color || 'bg-gray-500',
          description: '',
          isActive: true,
          userCount: 0,
          createdAt: new Date(),
          updatedAt: new Date()
        },
        role: data.role,
        permissions: data.permissions ? (typeof data.permissions === 'string' ? JSON.parse(data.permissions) : data.permissions) : [],
        isActive: Boolean(data.is_active),
        isOnline: Boolean(data.is_online),
        lastLoginAt: data.last_login_at ? new Date(data.last_login_at) : undefined,
        managerId: data.manager_id,
        totalTickets: data.total_tickets || 0,
        totalRevenue: data.total_revenue || 0,
        createdAt: new Date(data.created_at),
        updatedAt: new Date(data.updated_at)
      };

      setUser(userWithDates);
    } catch (error) {
      console.error('Error fetching user:', error);
      alert('Kullanıcı bilgileri yüklenirken bir hata oluştu');
    } finally {
      setLoading(false);
    }
  };

  const getRoleInfo = (role: string) => {
    const roleConfig: Record<string, { color: string; label: string }> = {
      system_admin: { color: 'bg-red-100 text-red-800', label: 'Sistem Yöneticisi' },
      agency_manager: { color: 'bg-purple-100 text-purple-800', label: 'Acente Yöneticisi' },
      department_manager: { color: 'bg-blue-100 text-blue-800', label: 'Departman Yöneticisi' },
      senior_specialist: { color: 'bg-green-100 text-green-800', label: 'Kıdemli Uzman' },
      specialist: { color: 'bg-indigo-100 text-indigo-800', label: 'Uzman' },
      assistant: { color: 'bg-gray-100 text-gray-800', label: 'Uzman Yardımcısı' },
      intern: { color: 'bg-yellow-100 text-yellow-800', label: 'Stajyer' }
    };

    return roleConfig[role] || roleConfig.specialist;
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!user) {
    return (
      <div className="p-6">
        <div className="text-center">
          <h2 className="text-2xl font-bold text-gray-900">Kullanıcı bulunamadı</h2>
          <Link href="/users" className="text-blue-600 hover:underline mt-4 inline-block">
            Kullanıcılara geri dön
          </Link>
        </div>
      </div>
    );
  }

  const roleInfo = getRoleInfo(user.role);

  return (
    <div className="p-6">
      {/* Header */}
      <div className="mb-6">
        <Link
          href="/users"
          className="inline-flex items-center text-sm text-gray-600 hover:text-gray-900 mb-4"
        >
          <ArrowLeft className="h-4 w-4 mr-1" />
          Kullanıcılara geri dön
        </Link>

        <div className="flex items-center justify-between">
          <div className="flex items-center">
            <div className="flex-shrink-0 h-16 w-16 rounded-full bg-gradient-to-r from-blue-600 to-purple-600 flex items-center justify-center mr-4">
              {user.avatar ? (
                <img src={user.avatar} alt={user.firstName} className="h-16 w-16 rounded-full" />
              ) : (
                <span className="text-xl font-medium text-white">
                  {user.firstName.charAt(0)}{user.lastName.charAt(0)}
                </span>
              )}
            </div>
            <div>
              <div className="flex items-center">
                <h1 className="text-3xl font-bold text-gray-900">{user.firstName} {user.lastName}</h1>
                {user.isOnline && (
                  <div className="ml-3 h-3 w-3 bg-green-500 rounded-full"></div>
                )}
              </div>
              <p className="text-gray-500 mt-1">{user.email}</p>
              <span className={`inline-flex items-center px-2 py-1 text-xs font-semibold rounded-full mt-2 ${roleInfo.color}`}>
                {roleInfo.label}
              </span>
            </div>
          </div>

          <div className="flex items-center space-x-3">
            <Link
              href={`/users/${user.id}/edit`}
              className="inline-flex items-center px-4 py-2 bg-white border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
            >
              <Edit className="h-4 w-4 mr-2" />
              Düzenle
            </Link>
          </div>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
        <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Activity className="h-8 w-8 text-green-600" />
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-500">Durum</p>
              <p className="text-2xl font-semibold text-gray-900">
                {user.isActive ? 'Aktif' : 'Pasif'}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <TrendingUp className="h-8 w-8 text-blue-600" />
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-500">Toplam Bilet</p>
              <p className="text-2xl font-semibold text-gray-900">{user.totalTickets || 0}</p>
            </div>
          </div>
        </div>

        <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <TrendingUp className="h-8 w-8 text-purple-600" />
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-500">Toplam Gelir</p>
              <p className="text-lg font-semibold text-gray-900">
                {(user.totalRevenue || 0).toLocaleString()} TRY
              </p>
            </div>
          </div>
        </div>

        <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
          <div className="flex items-center">
            <Calendar className="h-8 w-8 text-orange-600" />
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-500">Son Giriş</p>
              <p className="text-sm font-semibold text-gray-900">
                {user.lastLoginAt ? user.lastLoginAt.toLocaleDateString('tr-TR') : 'Hiç'}
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* User Info */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
          <h2 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Info className="h-5 w-5 mr-2" />
            Kişisel Bilgiler
          </h2>

          <dl className="space-y-4">
            <div>
              <dt className="text-sm font-medium text-gray-500 flex items-center">
                <Mail className="h-4 w-4 mr-2" />
                E-posta
              </dt>
              <dd className="mt-1 text-sm text-gray-900">{user.email}</dd>
            </div>

            {user.phone && (
              <div>
                <dt className="text-sm font-medium text-gray-500 flex items-center">
                  <Phone className="h-4 w-4 mr-2" />
                  Telefon
                </dt>
                <dd className="mt-1 text-sm text-gray-900">{user.phone}</dd>
              </div>
            )}

            <div>
              <dt className="text-sm font-medium text-gray-500 flex items-center">
                <Building2 className="h-4 w-4 mr-2" />
                Departman
              </dt>
              <dd className="mt-1">
                <div className="flex items-center">
                  <div className={`w-3 h-3 rounded-full ${user.department.color} mr-2`}></div>
                  <span className="text-sm text-gray-900">{user.department.name}</span>
                </div>
              </dd>
            </div>

            <div>
              <dt className="text-sm font-medium text-gray-500 flex items-center">
                <Shield className="h-4 w-4 mr-2" />
                Rol
              </dt>
              <dd className="mt-1">
                <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${roleInfo.color}`}>
                  {roleInfo.label}
                </span>
              </dd>
            </div>

            <div>
              <dt className="text-sm font-medium text-gray-500">Pozisyon</dt>
              <dd className="mt-1 text-sm text-gray-900">{user.position}</dd>
            </div>
          </dl>
        </div>

        <div className="bg-white p-6 rounded-lg shadow-sm border border-gray-200">
          <h2 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Calendar className="h-5 w-5 mr-2" />
            Hesap Bilgileri
          </h2>

          <dl className="space-y-4">
            <div>
              <dt className="text-sm font-medium text-gray-500">Kullanıcı ID</dt>
              <dd className="mt-1 text-sm text-gray-900 font-mono">{user.id}</dd>
            </div>

            <div>
              <dt className="text-sm font-medium text-gray-500">Hesap Durumu</dt>
              <dd className="mt-1">
                <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                  user.isActive
                    ? 'bg-green-100 text-green-800'
                    : 'bg-red-100 text-red-800'
                }`}>
                  {user.isActive ? 'Aktif' : 'Pasif'}
                </span>
              </dd>
            </div>

            <div>
              <dt className="text-sm font-medium text-gray-500">Çevrimiçi Durum</dt>
              <dd className="mt-1">
                <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                  user.isOnline
                    ? 'bg-green-100 text-green-800'
                    : 'bg-gray-100 text-gray-800'
                }`}>
                  {user.isOnline ? 'Çevrimiçi' : 'Çevrimdışı'}
                </span>
              </dd>
            </div>

            <div>
              <dt className="text-sm font-medium text-gray-500">Kayıt Tarihi</dt>
              <dd className="mt-1 text-sm text-gray-900">
                {user.createdAt.toLocaleString('tr-TR')}
              </dd>
            </div>

            <div>
              <dt className="text-sm font-medium text-gray-500">Son Güncelleme</dt>
              <dd className="mt-1 text-sm text-gray-900">
                {user.updatedAt.toLocaleString('tr-TR')}
              </dd>
            </div>

            {user.lastLoginAt && (
              <div>
                <dt className="text-sm font-medium text-gray-500">Son Giriş</dt>
                <dd className="mt-1 text-sm text-gray-900">
                  {user.lastLoginAt.toLocaleString('tr-TR')}
                </dd>
              </div>
            )}
          </dl>
        </div>
      </div>

      {/* Permissions */}
      {user.permissions && user.permissions.length > 0 && (
        <div className="mt-6 bg-white p-6 rounded-lg shadow-sm border border-gray-200">
          <h2 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Shield className="h-5 w-5 mr-2" />
            Yetkiler ({user.permissions.length})
          </h2>

          <div className="flex flex-wrap gap-2">
            {user.permissions.map((permission, index) => (
              <span
                key={index}
                className="inline-flex items-center px-3 py-1 text-sm font-medium rounded-full bg-blue-100 text-blue-800"
              >
                {permission}
              </span>
            ))}
          </div>
        </div>
      )}
    </div>
  );
}
